defmodule Reseau.Client do
  @moduledoc """
  Un serveur générique contenant un client.
  """

  use GenServer, restart: :temporary

  require Logger

  @doc """
  Création du processus.
  """
  def start_link(socket, options \\ []) do
    GenServer.start_link(__MODULE__, socket, options)
  end

  @impl true
  def init(socket) do
    {:ok, _} =
      Task.Supervisor.start_child(
        Reseau.MessengerSupervisor,
        Reseau.Messager,
        :démarrer,
        [self(), socket]
      )

    joueur = %Reseau.Joueur{}
    partie = %Reseau.Partie{}
    étape = Reseau.Etape.NomJoueur
    {joueur, partie} = étape.entrer(joueur, partie)
    afficher_étape(socket, étape, joueur, partie, "")
    {:ok, {socket, étape, joueur, partie}}
  end

  @impl true
  def handle_cast({:message, message}, {socket, étape, joueur, partie}) do
    message = String.trim(message)
    {étape, joueur, partie} =
      étape.gérer_commandes(joueur, partie, message)
      |> gérer_retour(socket, étape, joueur, partie)

    Logger.info("J'ai bien reçu le message: #{message}")
    {:noreply, {socket, étape, joueur, partie}}
  end

  @impl true
  def handle_info(:actualiser, {socket, étape, joueur, partie}) do
    afficher_étape(socket, étape, joueur, partie, "")
    {:noreply, {socket, étape, joueur, partie}}
  end

  defp gérer_retour(:silence, _socket, étape, joueur, partie), do: {étape, joueur, partie}

  defp gérer_retour(:prompt, socket, étape, joueur, partie) do
    afficher_étape(socket, étape, joueur, partie, "", true)
    {étape, joueur, partie}
  end

  defp gérer_retour(:rafraîchir, socket, étape, joueur, partie) do
    afficher_étape(socket, étape, joueur, partie, "", false)
    {étape, joueur, partie}
  end

  defp gérer_retour({:silence, message}, socket, étape, joueur, partie) do
    :gen_tcp.send(socket, message <> "\n")
    {étape, joueur, partie}
  end

  defp gérer_retour({:prompt, message}, socket, étape, joueur, partie) do
    afficher_étape(socket, étape, joueur, partie, message, true)
    {étape, joueur, partie}
  end

  defp gérer_retour({:rafraîchir, message}, socket, étape, joueur, partie) do
    afficher_étape(socket, étape, joueur, partie, message, false)
    {étape, joueur, partie}
  end

  defp gérer_retour({:rediriger, étape, joueur, partie}, socket, _étape, _joueur, _partie) do
    {joueur, partie} = étape.entrer(joueur, partie)
    afficher_étape(socket, étape, joueur, partie, "", false)
    {étape, joueur, partie}
  end

  defp gérer_retour({:rediriger, étape, joueur, partie, message}, socket, _étape, _joueur, _partie) when is_binary(message) do
    {joueur, partie} = étape.entrer(joueur, partie)
    afficher_étape(socket, étape, joueur, partie, message, false)
    {étape, joueur, partie}
  end

  defp gérer_retour({:rediriger, étape, joueur, partie, :silence}, _socket, _étape, _joueur, _partie) do
    {joueur, partie} = étape.entrer(joueur, partie)
    {étape, joueur, partie}
  end

  defp gérer_retour({:rediriger, étape, joueur, partie, :silence, message}, socket, _étape, _joueur, _partie) do
    {joueur, partie} = étape.entrer(joueur, partie)
    :gen_tcp.send(socket, message <> "\r\n")
    {étape, joueur, partie}
  end

  defp afficher_étape(socket, étape, joueur, partie, message, seulement_prompt \\ false) do
    titre = étape.titre(joueur, partie)
    texte = étape.texte(joueur, partie)
    prompt = étape.prompt(joueur, partie)

    complet = """
    #{(seulement_prompt && "") || titre}

    #{(seulement_prompt && "") || String.trim(texte)}
    #{message}
    #{prompt}
    """

    :gen_tcp.send(socket, String.trim(complet))
  end
end
